// (c) 1999 - 2025 OneSpan North America Inc. All rights reserved.


/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import MSSOrchestration

enum UIUtils {
    static func displayAlert(for error: OrchestrationError, on controller: UIViewController, dismissAction: (() -> Void)? = nil) {
        let parameters = getParameters(from: error)
        displayAlert(controller: controller, title: parameters.title, message: parameters.message, dismissAction: dismissAction)
    }
    
    static func displayAlert(controller: UIViewController, title: String, message: String, dismissAction: (() -> Void)? = nil) {
        displayAlert(controller: controller, title: title, message: message, firstButton: "OK", firstAction: dismissAction, secondButton: nil)
    }
    
    static func displayAlert(controller: UIViewController, title: String, message: String, firstButton: String?, firstAction: (() -> Void)? = nil, secondButton: String?, secondAction: (() -> Void)? = nil) {
        let alert = UIAlertController(title: title, message: message, preferredStyle: UIAlertController.Style.alert)
        
        if firstButton != nil {
            addButtonToAlert(alert, button: firstButton, dismissAction: firstAction)
        }
        if secondButton != nil {
            addButtonToAlert(alert, button: secondButton, dismissAction: secondAction)
        }
        
        controller.present(alert, animated: true, completion: nil)
    }
    
    static func displayProgress(controller: UIViewController, message: String) -> UIView {
        let view = UIView(frame: controller.view.frame)
        view.backgroundColor = UIColor(white: 0, alpha: 0.7)
        
        let indicator = UIActivityIndicatorView(style: UIActivityIndicatorView.Style.large)
        
        view.addSubview(indicator)
        setupConstraints(ofIndicator: indicator, in: view)
        
        indicator.startAnimating()
        indicator.center = view.center
        
        let label = UILabel()
        label.text = message
        label.textColor = .white
        label.numberOfLines = 0
        label.textAlignment = .center
        label.lineBreakMode = .byWordWrapping
        
        view.addSubview(label)
        setupConstraints(ofLabel: label, in: view, indicator: indicator)
        
        indicator.translatesAutoresizingMaskIntoConstraints = false
        label.translatesAutoresizingMaskIntoConstraints = false
        view.translatesAutoresizingMaskIntoConstraints = false
        
        controller.view.addSubview(view)
        
        setupConstraints(ofControllerView: controller.view, in: view)

        return view
    }
    
    private static func setupConstraints(ofIndicator indicator: UIView, in view: UIView) {
        view.addConstraint(
            NSLayoutConstraint(
                item: view,
                attribute: .centerX,
                relatedBy: .equal,
                toItem: indicator,
                attribute: .centerX,
                multiplier: 1,
                constant: 0
            )
        )
        view.addConstraint(
            NSLayoutConstraint(
                item: view,
                attribute: .centerY,
                relatedBy: .equal,
                toItem: indicator,
                attribute: .centerY,
                multiplier: 1,
                constant: 0
            )
        )
    }
    
    private static func setupConstraints(ofLabel label: UIView, in view: UIView, indicator: UIView) {
        view.addConstraint(
            NSLayoutConstraint(
                item: view,
                attribute: .centerX,
                relatedBy: .equal,
                toItem: label,
                attribute: .centerX,
                multiplier: 1,
                constant: 0
            )
        )
        view.addConstraint(
            NSLayoutConstraint(
                item: indicator,
                attribute: .bottom,
                relatedBy: .equal,
                toItem: label,
                attribute: .top,
                multiplier: 1,
                constant: -10
            )
        )
    }
    
    private static func setupConstraints(ofControllerView controllerView: UIView, in view: UIView) {
        controllerView.addConstraint(
            NSLayoutConstraint(
                item: controllerView,
                attribute: .leading,
                relatedBy: .equal,
                toItem: view,
                attribute: .leading,
                multiplier: 1,
                constant: 0
            )
        )
        controllerView.addConstraint(
            NSLayoutConstraint(
                item: controllerView,
                attribute: .trailing,
                relatedBy: .equal,
                toItem: view,
                attribute: .trailing,
                multiplier: 1,
                constant: 0
            )
        )
        controllerView.addConstraint(
            NSLayoutConstraint(
                item: controllerView,
                attribute: .top,
                relatedBy: .equal,
                toItem: view,
                attribute: .top,
                multiplier: 1,
                constant: 0
            )
        )
        controllerView.addConstraint(
            NSLayoutConstraint(
                item: controllerView,
                attribute: .bottom,
                relatedBy: .equal,
                toItem: view,
                attribute: .bottom,
                multiplier: 1,
                constant: 0
            )
        )
    }
    
    static func hideProgress(_ dialog: UIView?) {
        DispatchQueue.main.async {
            dialog?.removeFromSuperview()
        }
    }
    
    private static func getParameters(from error: OrchestrationError) -> (title: String, message: String) {
        let title = error.errorDescription ?? "Oops"
        let message = error.failureReason ?? "An error happened"
        let errorMessage = "\(message) - code \(error.errorCode)"
        
        return (title, errorMessage)
    }
    
    private static func addButtonToAlert(_ alert: UIAlertController, button: String?, dismissAction: (() -> Void)? = nil) {
        let action = UIAlertAction(title: button, style: UIAlertAction.Style.default) { _ in
            dismissAction?()
        }
        
        alert.addAction(action)
    }
}
